<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Binate Digital Payment Terminal</title>

    <script src="https://js.braintreegateway.com/web/dropin/1.39.0/js/dropin.min.js"></script>
    <script src="{{ asset('assets/js/sweetalert2@11.min.js') }}"></script>

    <meta name="csrf-token" content="{{ csrf_token() }}">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <link rel="stylesheet" href="{{ asset('assets/css/style.css') }}" />
</head>

<body>

    <div class="container">

        <div class="header">
            <img src="{{ asset('assets/images/logo.png') }}">
        </div>

        <div class="body-content">
            <div class="title">Payment Terminal</div>
            <div class="subtitle">
                Securely save your payment method for recurring installments.
            </div>

            <div class="divider"></div>

            <div class="section-title">Contact & Billing Information</div>

            <div class="form-group">
                <input type="text" id="firstName" placeholder="Full Name">
                <div class="error" id="firstNameError"></div>
            </div>

            <div class="form-group">
                <input type="email" id="email" placeholder="Email Address">
                <div class="error" id="emailError"></div>
            </div>

            <div class="form-group plan-wrapper">
                <input type="text" id="planId" value="{{ $planId }}" readonly>
                <span class="lock-icon">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor"
                        stroke-width="2">
                        <rect x="3" y="11" width="18" height="11" rx="2" />
                        <path d="M7 11V7a5 5 0 0110 0v4" />
                    </svg>
                </span>
                <div class="error" id="planIdError"></div>
            </div>



            <div id="dropin-container"></div>

            <div class="authorization-box">
                <label class="authorization-label">
                    <input type="checkbox" id="authorizeCheckbox">
                    <div class="auth-text">
                        I authorize Binate Digital Inc to securely store my payment info and
                        initiate recurring monthly charges for my project installments
                        as outlined in my signed service agreement.
                    </div>
                </label>
                <div class="error" id="authorizeCheckboxError"></div>
            </div>

            <button id="submit-button" disabled>Securely Save Card</button>
        </div>
    </div>

    <script>
        let dropinInstance;
        let isProcessing = false;

        const submitBtn = document.getElementById('submit-button');
        const authorizeCheckbox = document.getElementById('authorizeCheckbox');

        authorizeCheckbox.addEventListener('change', function() {
            submitBtn.disabled = !this.checked;
        });

        fetch('/api/braintree/token')
            .then(res => res.json())
            .then(data => {
                return braintree.dropin.create({
                    authorization: data.clientToken,
                    container: '#dropin-container'
                });
            })
            .then(instance => {
                dropinInstance = instance;
            });

        function clearErrors() {
            document.querySelectorAll('.error').forEach(el => el.innerText = '');
            document.querySelectorAll('input').forEach(el => el.classList.remove('invalid'));
        }

        function showFieldError(fieldId, message) {
            const errorEl = document.getElementById(fieldId + 'Error');
            const inputEl = document.getElementById(fieldId);

            if (errorEl) errorEl.innerText = message;
            if (inputEl) inputEl.classList.add('invalid');
        }

        submitBtn.addEventListener('click', function() {

            if (isProcessing) return;

            clearErrors();

            const firstName = document.getElementById('firstName').value.trim();
            const email = document.getElementById('email').value.trim();
            const planId = document.getElementById('planId').value;
            const authorizeChecked = authorizeCheckbox.checked;

            let hasError = false;

            if (!firstName) {
                showFieldError('firstName', 'Full Name is required');
                hasError = true;
            }

            if (!email) {
                showFieldError('email', 'Email is required');
                hasError = true;
            } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
                showFieldError('email', 'Enter a valid email address');
                hasError = true;
            }

            if (!planId) {
                showFieldError('planId', 'Invalid or missing plan.');
                hasError = true;
            }

            if (!authorizeChecked) {
                document.getElementById('authorizeCheckboxError').innerText =
                    'You must authorize recurring charges to continue.';
                hasError = true;
            }

            if (hasError) return;


            if (!dropinInstance || !dropinInstance.isPaymentMethodRequestable()) {
                Swal.fire(
                    "Incomplete Card Details",
                    "Please fill in complete and valid card information.",
                    "error"
                );
                return;
            }

            isProcessing = true;
            submitBtn.disabled = true;

            Swal.fire({
                title: "Processing...",
                text: "Please wait while we activate your subscription.",
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading()
            });

            dropinInstance.requestPaymentMethod(function(err, payload) {

                if (err) {
                    isProcessing = false;
                    submitBtn.disabled = false;
                    Swal.close();
                    Swal.fire("Payment Error", err.message, "error");
                    return;
                }

                fetch('/api/braintree/checkout', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                        },
                        body: JSON.stringify({
                            firstName,
                            email,
                            planId,
                            paymentMethodNonce: payload.nonce
                        })
                    })
                    .then(res => res.json())
                    .then(data => {

                        isProcessing = false;
                        submitBtn.disabled = false;
                        Swal.close();

                        if (!data.success) {
                            Swal.fire("Error", data.message || "Subscription failed.", "error");
                            return;
                        }

                        Swal.fire({
                            icon: "success",
                            title: "Subscription Active!",
                            text: "Your subscription has been activated successfully."
                        });
                    })
                    .catch(() => {
                        isProcessing = false;
                        submitBtn.disabled = false;
                        Swal.close();
                        Swal.fire("Error", "Something went wrong. Please try again.", "error");
                    });

            });

        });
    </script>

</body>

</html>
